<?php

namespace App\Http\Controllers;

use App\Config;
use App\Gst;
use App\Movie;
use App\Package;
use App\PpvPurchase;
use App\TvSeries;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Log;
use Paystack;

class PaystackController extends Controller
{
    public function paystackgateway()
    {
        return Paystack::getAuthorizationUrl()->redirectNow();
    }

    public function paystackcallback()
    {
        $auth = Auth::user();
        $payment = Paystack::getPaymentData();
        $config = Config::findOrFail(1);
        $gst = Gst::first();
        $base_currency = $config->currency_code;

        if ($payment['data']['status'] === 'success' && $payment['status'] === 'true') {
            $payment_id = $payment['data']['reference'];
            $payment_amount = $payment['data']['amount'] / 100; // kobo to Naira
            $payment_method = 'paystack';
            $payment_status = 1;
            $type = $payment['data']['metadata']['type'] ?? null;

            if (!$type) {
                Log::error('Null payment type in metadata', ['payment' => $payment]);
                return redirect('/')->with('error', __('Invalid payment type.'));
            }

            if ($type === 'subscription') {
                $plan = Package::wherePlanId($payment['data']['metadata']['plan_id'])->first();
                $checkout = new SubscriptionController;
                return $checkout->subscribe(
                    $payment_id,
                    $payment_method,
                    $plan->id,
                    $payment_status,
                    $payment_amount
                );
            }

            if ($type === 'rental') {
                $movie_id = $payment['data']['metadata']['movie_id'] ?? null;
                if (!$movie_id) {
                    Log::error('Null movie_id for rental', ['metadata' => $payment['data']['metadata']]);
                    return redirect('/')->with('error', __('Invalid movie ID'));
                }

                $movie = Movie::findOrFail($movie_id);
                if (!$movie->hours || !$movie->slug) {
                    Log::error('Null movie hours or slug', [
                        'movie_id' => $movie_id,
                        'hours' => $movie->hours,
                        'slug' => $movie->slug
                    ]);
                    return redirect('/')->with('error', __('Invalid movie data'));
                }

                $converted_price = Session::has('current_currency')
                    ? round(currency($movie->price, $from = $base_currency ?? 'INR', $to = ucfirst(Session::get('current_currency')), $format = false), 2)
                    : round($movie->price, 2);

                $converted_offer_price = null;
                if (isset($movie->offer_price) && $movie->offer_price > 0) {
                    $converted_offer_price = Session::has('current_currency')
                        ? round(currency($movie->offer_price, $from = $base_currency ?? 'INR', $to = ucfirst(Session::get('current_currency')), $format = false), 2)
                        : round($movie->offer_price, 2);
                }

                $base_price = $converted_offer_price ?? $converted_price;

                $gst_amount = 0;
                $total_with_gst = $base_price;
                if ($gst && $gst->gst_enable && isset($gst->gst_per) && is_numeric($gst->gst_per)) {
                    $gst_amount = ($base_price * $gst->gst_per) / 100;
                    $total_with_gst = $base_price + $gst_amount;
                } else {
                    Log::info('GST disabled or null', [
                        'gst_enable' => $gst->gst_enable ?? 'null',
                        'gst_per' => $gst->gst_per ?? 'null'
                    ]);
                }

                Log::info('Movie rental payment details', [
                    'movie_id' => $movie->id,
                    'movie_title' => $movie->title,
                    'base_price' => $base_price,
                    'gst_percentage' => $gst ? ($gst->gst_enable ? $gst->gst_per : 0) : 0,
                    'gst_amount' => $gst_amount,
                    'total_with_gst' => $total_with_gst,
                    'payment_amount' => $payment_amount,
                    'converted_price' => $converted_price,
                    'converted_offer_price' => $converted_offer_price,
                    'current_currency' => Session::get('current_currency') ?? ($base_currency ?? 'INR')
                ]);

                if (abs($payment_amount - $total_with_gst) > 0.01) {
                    Log::error('Payment amount mismatch for movie rental', [
                        'expected' => $total_with_gst,
                        'received' => $payment_amount,
                        'difference' => $total_with_gst - $payment_amount
                    ]);

                    if (abs($payment_amount - $total_with_gst) > 1.00) {
                        return redirect()->back()->with('error', __('Payment amount mismatch. Please try again.'));
                    }
                }

                $purchase = PpvPurchase::create([
                    'user_id' => auth()->id(),
                    'movie_id' => $movie->id,
                    'payment_id' => $payment_id,
                    'gst' => $gst ? ($gst->gst_enable ? $gst->gst_per : 0) : 0,
                    'price' => $converted_price,
                    'offer_price' => $converted_offer_price ?? $converted_price,
                    'hours' => $movie->hours,
                    'expires_at' => now()->addHours($movie->hours),
                    'type' => 'movie',
                    'payment_type' => 'Paystack',
                    'status' => 1,
                ]);

                Log::info('PPV purchase created for movie', ['purchase_id' => $purchase->id]);

                $route = getSubscription()->getData()->subscribed
                    ? 'movie/detail/'
                    : 'movie/guest/detail/';
                return redirect()->to($route . $movie->slug)->with('success', __('Movie purchase successful!'));
            }

            if ($type === 'rental_series') {
                $series_id = $payment['data']['metadata']['series_id'] ?? null;
                if (!$series_id) {
                    Log::error('Null series_id for rental_series', ['metadata' => $payment['data']['metadata']]);
                    return redirect('/')->with('error', __('Invalid season ID'));
                }

                $series = Season::findOrFail($series_id);
                if (!$series->hours || !$series->season_slug) {
                    Log::error('Null series hours or season_slug', [
                        'series_id' => $series_id,
                        'hours' => $series->hours,
                        'season_slug' => $series->season_slug
                    ]);
                    return redirect('/')->with('error', __('Invalid season data'));
                }

                $converted_price = Session::has('current_currency')
                    ? round(currency($series->price, $from = $base_currency ?? 'INR', $to = ucfirst(Session::get('current_currency')), $format = false), 2)
                    : round($series->price, 2);

                $converted_offer_price = null;
                if (isset($series->offer_price) && $series->offer_price > 0) {
                    $converted_offer_price = Session::has('current_currency')
                        ? round(currency($series->offer_price, $from = $base_currency ?? 'INR', $to = ucfirst(Session::get('current_currency')), $format = false), 2)
                        : round($series->offer_price, 2);
                }

                $base_price = $converted_offer_price ?? $converted_price;

                $gst_amount = 0;
                $total_with_gst = $base_price;
                if ($gst && $gst->gst_enable && isset($gst->gst_per) && is_numeric($gst->gst_per)) {
                    $gst_amount = ($base_price * $gst->gst_per) / 100;
                    $total_with_gst = $base_price + $gst_amount;
                } else {
                    Log::info('GST disabled or null', [
                        'gst_enable' => $gst->gst_enable ?? 'null',
                        'gst_per' => $gst->gst_per ?? 'null'
                    ]);
                }

                Log::info('Season rental payment details', [
                    'season_id' => $series->id,
                    'season_title' => 'Season ' . $series->season_no,
                    'base_price' => $base_price,
                    'gst_percentage' => $gst ? ($gst->gst_enable ? $gst->gst_per : 0) : 0,
                    'gst_amount' => $gst_amount,
                    'total_with_gst' => $total_with_gst,
                    'payment_amount' => $payment_amount,
                    'converted_price' => $converted_price,
                    'converted_offer_price' => $converted_offer_price,
                    'current_currency' => Session::get('current_currency') ?? ($base_currency ?? 'INR')
                ]);

                if (abs($payment_amount - $total_with_gst) > 0.01) {
                    Log::error('Payment amount mismatch for season rental', [
                        'expected' => $total_with_gst,
                        'received' => $payment_amount,
                        'difference' => $total_with_gst - $payment_amount
                    ]);

                    if (abs($payment_amount - $total_with_gst) > 1.00) {
                        return redirect()->back()->with('error', __('Payment amount mismatch. Please try again.'));
                    }
                }

                $purchase = PpvPurchase::create([
                    'user_id' => auth()->id(),
                    'season_id' => $series->id,
                    'payment_id' => $payment_id,
                    'gst' => $gst ? ($gst->gst_enable ? $gst->gst_per : 0) : 0,
                    'price' => $converted_price,
                    'offer_price' => $converted_offer_price ?? $converted_price,
                    'hours' => $series->hours,
                    'expires_at' => now()->addHours($series->hours),
                    'type' => 'series',
                    'payment_type' => 'Paystack',
                    'status' => 1,
                ]);

                Log::info('PPV purchase created for season', ['purchase_id' => $purchase->id]);

                $route = getSubscription()->getData()->subscribed
                    ? 'show/detail/'
                    : 'show/guest/detail/';
                return redirect()->to($route . $series->season_slug)->with('success', __('Series rental successful!'));
            }

            return redirect('/')->with('error', __('Invalid payment type.'));
        }

        return redirect('/')->with('error', __('Payment failed.'));
    }
}